/**************************************************************************
 *
 * Copyright 2010, 2011 BMW Car IT GmbH 
 * Copyright (C) 2011 DENSO CORPORATION and Robert Bosch Car Multimedia Gmbh
 *
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 ****************************************************************************/
#include "egl_helper.h"
#include "ilm_control.h"
#include <stdio.h>
#include <sys/time.h>

t_ilm_bool createContext(t_ilm_layer layerid, t_ilm_surface surfaceid,
                         t_ilm_int width, t_ilm_int height)
{
    ilmErrorTypes error = ILM_SUCCESS;
    t_ilm_nativehandle nativehandle = 0;

    do
    {
        error = ilm_clientUtil_init();
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to init ilmClientUtil.\n");
            break;
        }

        /* create Pixmap one time */
        error = ilm_createNativePixmap(width, height);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to create native pixmap.\n");
            break;
        }
        printf("native pixmap=0x%08lx\n", (unsigned long)ilm_getNativeWindow());
        ilm_destroyNativePixmap();

        error = ilm_createNativeWindow(width, height);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to create native window.\n");
            break;
        }

        error = ilm_createEGLContext();
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to create egl context.\n");
            break;
        }

        nativehandle = ilm_getNativeHandle();
        if (0 == nativehandle)
        {
            printf("Error: failed to get nativehandle.\n");
            break;
        }

        printf("native display=0x%08lx\n", (unsigned long)ilm_getNativeDisplay());
        printf("native window=0x%08lx\n", (unsigned long)ilm_getNativeWindow());
        printf("native handle=0x%08lx\n", (unsigned long)nativehandle);

        error = ilm_layerCreateWithDimension(&layerid, width, height);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to create layer.\n");
            break;
        }

        error = ilm_surfaceCreate(nativehandle, width, height,
                                  ILM_PIXELFORMAT_RGBA_8888, &surfaceid);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to create surface.\n");
            break;
        }

        error = ilm_surfaceSetDestinationRectangle(surfaceid,
                                                   0, 0, width, height);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to set destination rectangle.\n");
            break;
        }

        error = ilm_surfaceSetSourceRectangle(surfaceid,
                                              0, 0, width, height);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to set source rectangle.\n");
            break;
        }

        error = ilm_layerAddSurface(layerid, surfaceid);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to set source rectangle.\n");
            break;
        }

        error = ilm_layerSetVisibility(layerid, ILM_TRUE);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to set layer visibility ON.\n");
            break;
        }

        error = ilm_surfaceSetVisibility(surfaceid, ILM_TRUE);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to set surface visibility ON.\n");
            break;
        }

        error = ilm_displaySetRenderOrder(0, &layerid, 1);
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to set screen render order.\n");
            break;
        }

        error = ilm_commitChanges();
        if (ILM_SUCCESS != error)
        {
            printf("Error: failed to commit changes.\n");
            break;
        }

        return ILM_TRUE;
    } while(0);

    destroyContext(layerid, surfaceid);
    return ILM_FALSE;
}

void destroyContext(t_ilm_layer layerid, t_ilm_surface surfaceid)
{
    ilm_surfaceRemove(surfaceid);
    ilm_layerRemove(layerid);
    ilm_destroyEGLContext();
    ilm_destroyNativeWindow();
    ilm_clientUtil_destroy();
}

t_ilm_uint GetTickCount()
{
    struct timeval ts;
    gettimeofday(&ts, 0);
    return (t_ilm_uint) (ts.tv_sec * 1000 + (ts.tv_usec / 1000));
}

void swapBuffers()
{
    ilm_swapBuffers();
}
